function L = membrane(k,M,N,NP)
%	MEMBRANE(k,M,N,NP) returns the k-th eigenfunction of the L-shaped
%	membrane, k <= 12.  See the covers of various MathWorks publications.
%
%	  M = number of points on 1/3 of boundary, recommend M = 15.
%	  N = number of terms in sum, recommend N = 9.
%	  NP = number of terms in partial sum, recommend NP = N.
%
%	Use MEMBRANE(1,15,9,2) to generate MathWorks logos.

%	Out-of-date reference: Fox, Henrici & Moler, SIAM J. Numer. Anal. 4,
%	1967, pp.89-102.

%	Cleve Moler 4-21-85, 7-21-87.
%	Copyright (c) 1985, 1986, 1987 by the MathWorks, Inc.

if nargin < 4
	NP = N;  % Number of terms in partial sum.
end
lambda = [9.6397238445, 15.19725192, 2*pi^2, 29.5214811, 31.9126360, ...
    41.4745099, 44.948488, 5*pi^2, 5*pi^2, 56.709610, 65.376535, 71.057755];
lambda = lambda(k);
sym = [1 2 3 2 1 1 2 3 3 1 2 1];
sym = sym(k);

% Part of the boundary in polar coordinates
theta = (1:3*M)'/M * pi/4;
r = [ones(M,1)./cos(theta(1:M)); ones(2*M,1)./sin(theta(M+1:3*M))];

% if sym = 1, alfa = [1 5 7 11 13 17 19 ... ] * 2/3, (odd, not divisible by 3)
% if sym = 2, alfa = [2 4 8 10 14 16 20 ... ] * 2/3, (even, not divisible by 3)
% if sym = 3, alfa = [3 6 9 12 15 18 21 ... ] * 2/3, (multiples of 3)
alfa = sym;
del = sym + 3*(sym == 1);
for j = 2:N,
   alfa(j) = alfa(j-1) + del;
   del = 6-del;
end;
alfa = alfa' * 2/3;

% Build up the matrix of fundamental solutions evaluated on the boundary.
A = [];
t = sqrt(lambda)*r;
for j = 1:N,
   A(:,j) = bessela(alfa(j),t).*sin(alfa(j).*theta);
end

% The desired coefficients are the null vector.
% (lambda was chosen so that the matrix is rank deficient).  
[Q,R,E] = qr(A);
k = 1:N-1;
c = -R(k,k)\R(k,N);
c(N) = 1;
c = E*c(:);
if c(1) < 0, c = -c; end
% The residual should be small.
% res = norm(A*c,inf)/norm(abs(A)*abs(c),inf)

% Now evaluate the eigenfunction on a rectangular mesh in the interior.
MM = 2*M+1;
x = ones(M+1,1)*(-M:M)/M;
y = (M:-1:0)'/M*ones(1,MM);
r = sqrt(x.*x + y.*y);
theta = atan2(y,x);
theta(M+1,M+1) = 0;
S = zeros(M+1,MM);
r = sqrt(lambda)*r;
for j = 1:NP
   S = S + c(j) * bessela(alfa(j),r) .* sin(alfa(j)*theta);
   L = zeros(MM,MM);
   L(1:M+1,:) = triu(S);
   if sym == 1, L = L + L' - diag(diag(L)); end
   if sym == 2, L = L - L'; end
   if sym == 3, L = L + L' - diag(diag(L)); end
   mesh(L)
end

% The following two lines are commented out. They zero the edges.
% edge = zeros(MM,1);
% L(:,1) = edge; L(:,MM) = edge; L(1,:) = edge'; L(MM,:) = edge';

L = rot90(L,2);
