function k = ellipk(kind,m)
%ELLIPK Complete elliptic integrals.
%	ELLIPK(KIND,M) returns the value of the complete elliptic
%	integral of the kind KIND, evaluated at M.  As currently
%	implemented, M is limited to 0 < M < 1 and KIND must be 1.
%	The accuracy of ELLIPK(KIND,M) is EPS.
%
%	Be sure you don't confuse the modulus K with the parameter M -
%	they are related in the following way:  M = K^2

%	L. Shure 1-9-88
%	Copyright (c) 1988 by the MathWorks, Inc.

%	ELLIPK uses the method of the arithmetic-geometric mean
%	described in [1].

% References:
%	[1] M. Abramowitz and I.A. Stegun, "Handbook of Mathematical
%	    Functions" Dover Publications", 1965, 17.6.

%

if kind ~= 1
	error('ELLIPK currently does complete elliptic integrals of the first kind only.');
end
tol = eps(1);
[mm,nn] = size(m);
im = find(m >= 1-tol);
k = zeros(mm,nn);
m = m(:).';	% make row vector
mmax = mm*nn;
a0 = ones(1,mmax);
c0 = sqrt(m);
b0 = sqrt(1-m);
a = a0;
b = b0;
c = c0;
while any(abs(c) > tol)
	a0(:) = a;
	b0(:) = b;
	c0(:) = c;
	a(:) = .5*(a0+b0);
	b(:) = sqrt(a0.*b0);
	c(:) = .5*(a0-b0);
end
k(:) = pi ./ (2 .* a);
if ~isempty(im)
	if ieee
		disp('Warning: ELLIPK(1) is infinite.')
		k(im) = inf*ones(max(size(im)),1);
	else
		error('ELLIPK(1) is infinite.')
	end
end
