function [zi,weights]=interp3(x,y,z,xi,yi,weights)
%INTERP3 2-D biharmonic data interpolation (gridding).
%	ZI = INTERP3(X,Y,Z,XI,YI) uses biharmonic interpolation to
%	find a matrix ZI corresponding to new abscissa vectors XI
%	and YI, given matrix Z whose rows and columns correspond to
%	vectors X and Y.
%
%	If X, Y, and Z are all vectors, INTERP3 interpolates within the
%	data given by the (possibly irregularly spaced) ordered triplets
%	(X,Y,Z) to return matrix ZI.
%
%	ZI = INTERP3(Z,M,N) returns an M-by-N matrix ZI whose values
%	are interpolated within matrix Z.
%
%	[ZI,WEIGHTS]=INTERP3(X,Y,Z,XI,YI) also returns the computed
%	interpolation WEIGHTS, which can be reused as follows:
%	INTERP3(X,Y,Z,XI,YI,WEIGHTS) uses the given WEIGHTS vector,
%	instead of recalculating them from X, Y, and Z.

%	Copyright (c) The MathWorks, Inc., 1988-89
%	Charles R. Denham

%  INTERP3 called with no arguments interpolates and contours
%   some random (x,y,z) values.

% Reference:  David T. Sandwell, Biharmonic spline
%  interpolation of GEOS-3 and SEASAT altimeter
%  data, Geophysical Research Letters, 2, 139-142,
%  1987.  Describes interpolation using value or
%  gradient of value in any dimension.

if nargin < 1      % Random data for demonstration plot.
   z = rand(10,1);
   x = rand(10,1);
   y = rand(10,1);
   xi = ((0:10) ./ 10).';
   yi = xi;
end

if nargin == 3
   rows = y; cols = z;
   z = x;
   xi = (0:rows-1) ./ (rows-1); yi = (0:cols-1) ./ (cols-1);
   [m,n] = size(z);
   m = (0:m-1) ./ (m-1); n = (0:n-1) ./ (n-1);
   [x, y] = meshdom(m, n);
end

if prod(size(x)) ~= prod(size(z))
   if length(x(:)) .* length(y(:)) == prod(size(z))
      [x, y] = meshdom(x, y);
     else
      error('Inconsistent sizes for x, y, and z.')
   end
end

z = z(:);
x = x(:);
y = y(:);
m = length(x);

xy = x + y*sqrt(-1);

if nargin < 6
   xym = xy * ones(1,m);
   d = abs(xym - xym.');
   mask = (d == 0);
   d = d + mask;
   g = (d.^2) .* (log(d)-1);   % Green's function.
   g = g .* (1 - mask);
   weights = g \ z;
end
oldxi = xi;
oldyi = yi;
xi = xi(:);
yi = yi(:);
[xxi,yyi] = meshdom(xi.',yi.');
xi = xxi(:);
yi = yyi(:);
[im,in] = size(xi);
xym = ones(im,1) * xy.';
xiyi = xi + yi*sqrt(-1);
xiyim = xiyi * ones(1,m);
d = abs(xiyim - xym);
mask = (d == 0);
d = d + mask;
g = (d.^2) .* (log(d)-1);   % Green's function.
g = g .* (1 - mask);
temp = g * weights;
zi = xxi;
zi(:) = temp;

if nargin < 1          % Demonstration plot.
   contour(zi,10,oldxi,oldyi); hold on
   plot(x,y,'*'), hold off
   title('Demonstration of INTERP3')
   xlabel('x'), ylabel('y')
end
