function [S,X] = rat(X,LEN,dmax,str)
%RAT	Rational approximation.
%	RAT(X) returns a rational approximation to the elements of X
%	by removing the roundoff error from results that should be
%	"simple" rational numbers.
%	
%	[N,D] = RAT(X) returns two integer matrices N and D such
%	that N./D = RAT(X).
%	RAT(X,'S') returns a text matrix with the rational
%	approximation to X. 
%
%	For example, try RAT(HILB(4)).
%
%	The rational approximations are generated by truncating
%	continued fraction expansions.  
%	RAT(X,LEN,MAX) controls the length of the expansion and the
%	truncation level.  By default, LEN = 13, MAX = 10000.  See
%	the user's guide for more information.

%	C.B. Moler, 3-23-87.
%	Revised, 12-20-88 LS.
%	Copyright (c) 1987 by the MathWorks, Inc.

decrat = 0;	% do rational approximation into text matrix
if nargin < 2
	LEN = 13;
	dmax = 10000;
	decrat=1;
elseif nargin == 2
	if isstr(LEN)
		decrat = 0; dmax = 10000; LEN = 13;
	else
		decrat = 1; dmax = 10000;
	end
elseif nargin == 3
	if isstr(dmax)
		decrat = 0; dmax = 10000;
	else
		decrat = 1;
	end
end
if ~decrat & nargout > 1
	error('RAT may return text output only if there is 1 left-hand side.')
end

[m,n] = size(X);
S = [];
for i = 1:m
   s = [];
   for j = 1:n
      x = abs(X(i,j));
      d = fix(x);
      x = x - d;
      k = 1;
      p = 1/abs(dmax);
      while x > p
         x = 1/x;
         k = k+1;
         d = [d fix(x)];
         x = x - d(k);
         p = p*d(k);
      end
      if dmax < 0, d, end
      num = d(k);
      den = 1;
      for k = k-1:-1:1
         t = num;
         num = d(k)*num + den;
         den = t;
      end
      num = sign(X(i,j))*num;
      if decrat == 0
         sj = sprintf('%6.0f',num);
         if den ~= 1
            sj = [sj '/' sprintf('%.0f',den)];
         else
            sj = [' ' sj];
         end
         if (length(sj) > LEN) | (num==0 & x~=0)
            sj = sprintf('%13.3e',X(i,j));
         else
            sj = [sj blanks(LEN-length(sj))];
         end
         s = [s sj];
      else
         s = [s num];
      end
      X(i,j) = den;
   end
   S = [S; s];
end
if nargout == 1 & decrat == 1
	S = S ./ X;
end
