echo on
clc

%	 This demonstration uses census data from 1900-1980 to predict the
%	 population of the U.S. in 1990.  The example illustrates the
%	 dangers of using high order polynomial fits for extrapolation.

pause % Strike any key to continue.
clc

%	 We start by forming a time vector of the years from 1900-1980.
%	 One unit in t corresponds to 10 years.

t = (0:8)'


pause % Strike any key to continue.
clc

%	 Let p be the vector of census data.
%	 One unit in p is a million people.

p = [75.995 91.972 105.711 123.203 131.669 ...
     150.697 179.323 203.212 226.505]'

pause %  Let's plot the data.   Strike any key for plot.
echo off
axis([0 10 0 400]);
plot(t,p,'o')
title('Population of the U.S. 1900-1980')
xlabel('1900 - 2000'), ylabel('Millions'), pause
echo on
clc
%	 A polynomial fit to the data is a function of y(t) of the form:
%
%	              d       d-1
%	   y(t) = c1*t  + c2*t    + ... + cn ,
%
%	 The degree is d and the number of coefficients is n = d+1.
%	 There are 9 observations so the data can be fit exactly by 
%	 a polynomial with d = 8 and n = 9.  The coefficients 
%	 c1, c2, ... c9 are determined by solving a 9 by 9 system 
%	 of simultaneous linear equations:
%
%       		 A*c = p
%
%	 The columns of A are successive powers of the t vector.

pause % Strike any key to continue.
A=[];
clc

%	 Here is one way to form A

n = max(size(t));
for j = 1:n
	A(:,j) = t.^(n-j);
end

pause % Strike any key to see A.
format compact
clc
A	 % Here is A:
pause % Strike any key to continue.
format
clc
%	 The solution to the system of simultaneous linear equations
%		     A*c = p
%	 is obtained with the MATLAB "matrix division" operation:

c = A\p
%	 Notice that the leading coefficient is negative, so for large t,
%	 the polynomial y(t) is a decreasing function of t.

pause % Strike any key to continue.
clc

%	To find the population in 1990, evaluate the polynomial at t = 9.

pop90 = polyval(c,9)

%	 Hopefully, this is a severe UNDERestimate.


pause % Strike any key to continue.
clc

%	 Was the underestimate caused by roundoff error?
%	 Check the condition number of the matrix A.
%	 (This involves doing a 9 by 9 singular value decomposition.)

cond(A)

%	 This says we lost about 9 significant digits in computing the 
%	 coefficients in the polynomial.  But we have about 16 digits 
%	 to work with, so our coefficients, and our predicted population,
%	 are accurate to at least the number of digits we are printing.

pause % Strike any key to continue.
clc

%	 To see what went wrong, let's plot the polynomial.
%	 The vector x corresponds to all the years from 1900 to 2000 and the
%	 vector y contains the values of the polynomial at those 101 points.

x = (0: 0.1: 10)';
y = polyval(c,x);

pause % Strike any key for plot.

plot(t,p,'o',x,y,'-'), title('Polynomial interpolation, degree 8'), pause
clc

%	 The values obtained from the degree 8 polynomial decrease very
%	 rapidly outside the interval containing the data points.


pause % Strike any key to continue.
clc

%	 The polynomial actually goes through zero.  When?
%	 Find the roots of the polynomial.

z = roots(c)

pause % Strike any key to continue.
clc

%	 Most of the roots are complex.  But the first one is real
%	 and positive.  That's the one we're interested in.
%	 Convert to years and days.

time = 10*real(z(1))
year = fix(time);  day = 365*rem(time,1);  doomsday = [year day]

%	That's September 27, 1991!

pause % Strike any key to continue.
clc

%	Reduce the degree from 8 to 6.  There are still 9 equations, 
%	but now only 7 coefficients.  Drop the first two columns of A 
%	and find the best fit in a least square sense.

B = A(:,3:9);
c = B\p

%	 Now the leading coefficient is positive.


pause % Strike any key to continue.
clc

%	Predict the 1990 population with the 6th degree polynomial.

pop90 = polyval(c,9)

%	That doesn't look too bad.  Do you trust it?


pause %  Let's plot this second fit.   Strike any key for plot.
echo off
clc
y = polyval(c,x);
plot(t,p,'o',x,y,'-')
title('Least squares polynomial, degree 6')
pause
echo on
%	This illustrates the dangers of using polynomials of even
%	modest degree for extrapolation beyond the data interval.


pause % Strike any key to continue.
clc

%	Now we'll let you do your own polynomial fits to the data.
%	First of all, we'll prompt you for the degree of the polynomial.
%	The two fits you have already seen had degree 8 and degree 6.
%	Then we'll prompt you for something called the Levenberg-Marquardt
%	stabilization parameter.  The previous fits had no stabilization,
%	so the parameter was zero.  Low degree polynomials don't need much
%	stabilization, but, as we've seen, higher degree polynomials need
%	more.  The high order coefficient in the 8th degree polynomial is
%	quite sensitive to the parameter.  If you choose degree 8, try
%	stabilization parameters around 0.00005.  Other than that, nobody
%	really knows how to choose the Levenberg-Marquardt parameter,
%	so you're on your own.
%
%	We use the singular value decomposition to do polynomial fits with
%	Levenberg-Marquardt stabilization. If you know what this means, you
%	might be interested in the next page of information.  Otherwise
%	ignore it and just supply input data when asked.


pause % Strike any key to continue.
if exist('testall'), axis; return, end  % MathWorks diagnostics
clc

%	Polynomial fits with Levenberg-Marquardt stabilization.

while 1
   d = input('Enter polynomial degree. (Less than 9. Enter 0 to quit.) ');
   if isempty(d), break, end
   if (d <= 0) | (d >= 9), break, end
   [U,S,V] = svd(A(:,n-d:n));
   if d > 0, S = diag(S);, end;
   lambda = input('Enter stabilizing parameter. (Suggest .5 or 0.) ');
   if isempty(lambda), lambda = .1; end
   c = U(:,1:d+1)'*p;
   c = S./(S.*S + lambda).*c;
   c = V*c;
   pop90 = polyval(c,9)
   y = polyval(c,x);
   disp('Strike any key for plot')
   pause
   plot(t,p,'o',x,y,'-')
   pause
   clc
end
axis;
clc
