function [reout,im,w] = nyquist(a,b,c,d,iu,w)
%NYQUIST Nyquist frequency response for continuous-time linear systems.
%	NYQUIST(A,B,C,D,IU) produces a Nyquist plot from the single input
%	IU to all the outputs of the system:             
%               .                                    -1
%               x = Ax + Bu             G(s) = C(sI-A) B + D  
%               y = Cx + Du      RE(w) = real(G(jw)), IM(w) = imag(G(jw))
%
%	The frequency range and number of points are chosen automatically.
%
%	NYQUIST(NUM,DEN) produces the Nyquist plot for the polynomial 
%	transfer function G(s) = NUM(s)/DEN(s) where NUM and DEN contain
%	the polynomial coefficients in descending powers of s. 
%
%	NYQUIST(A,B,C,D,IU,W) or NYQUIST(NUM,DEN,W) uses the user-supplied
%	freq. vector W which must contain the frequencies, in radians/sec,
%	at which the Nyquist response is to be evaluated.  When invoked 
%	with left hand arguments,
%		[RE,IM,W] = NYQUIST(A,B,C,D,...)
%		[RE,IM,W] = NYQUIST(NUM,DEN,...) 
%	returns the frequency vector W and matrices RE and IM with as many
%       columns as outputs and length(W) rows.  No plot is drawn on the 
%	screen.
%	See also: LOGSPACE,MARGIN,BODE, and NICHOLS.

% 	J.N. Little 10-11-85
%	Revised A.C.W.Grace 8-15-89, 2-12-91
%	Revised Clay M. Thompson 7-9-90
%	Copyright (c) 1985-91 by the MathWorks, Inc.

if nargin==0, eval('exresp(''nyquist'')'), return, end

% --- Determine which syntax is being used ---
if (nargin==1),
  error('Wrong number of input arguments.');

elseif (nargin==2),	% Transfer function form without frequency vector
  num  = a; den = b; 
  w = freqint2(num,den,30);
  [ny,nn] = size(num); nu = 1;

elseif (nargin==3),	% Transfer function form with frequency vector
  num = a; den = b;
  w = c;
  [ny,nn] = size(num); nu = 1;

elseif (nargin==4),	% State space system, w/o iu or frequency vector
  error(abcdchk(a,b,c,d));
  w = freqint2(a,b,c,d,30);
  [iu,nargin,re,im]=mulresp('nyquist',a,b,c,d,w,nargout,0);
  if ~iu, if nargout, reout = re; end, return, end
  [ny,nu] = size(d);

elseif (nargin==5),	% State space system, with iu but w/o freq. vector
  error(abcdchk(a,b,c,d));
  w = freqint2(a,b,c,d,30);
  [ny,nu] = size(d);

else
  error(abcdchk(a,b,c,d));
  [ny,nu] = size(d);

end

if nu*ny==0, im=[]; w=[]; if nargout~=0, reout=[]; end, return, end

% Compute frequency response
if (nargin==2)|(nargin==3)
  g = freqresp(num,den,sqrt(-1)*w);
else
  g = freqresp(a,b,c,d,iu,sqrt(-1)*w);
end
re=real(g); 
im=imag(g);

% If no left hand arguments then plot graph.
if nargout==0,
  status = inquire('hold');
%  if ~status, clg, end

  % Make arrows
  sx=max(re)-min(re);   [sy,sample]=max(abs(2*im));
  arrow=[-1;0;-1]*sx/50+sqrt(-1)*[1;0;-1]*sy/50;
  sample=sample+(sample==1);
  reim=diag(g(sample,:));
  d=diag(g(sample+1,:)-g(sample-1,:)); 
  rot = sign(d);  % Arrow is always horizonatl i.e. -1 or +1
  % rot=d./abs(d)  Use this when arrow is not horizontal
  xy =ones(3,1)*reim' + ones(3,1)*rot'.*arrow;
  xy2=ones(3,1)*reim' - ones(3,1)*rot'.*arrow;
  [m,n]=size(g); 
  if (n==1) 
  	plot(re,im,'r-',re,-im,'g--', real(xy),-imag(xy),'r-',real(xy2),imag(xy2),'g-');
  else
  	plot(re,im,'-',re,-im,'--')
	hold on
  	plot(real(xy),-imag(xy),'-',real(xy2),imag(xy2),'-')
  end
  hold on
  xlabel('Real Axis'), ylabel('Imag Axis')

  % Make cross at s = -1 + j0
  [s,limits] = inquire('axis');
% Axis
  plot([limits(1:2);0,0]',[0,0;limits(3:4)]','w:');
% -1 point
  line1 = (limits(2)-limits(1))/50;
  line2 = (limits(4)-limits(3))/50;
  plot([-1+line1, -1-line1], [0,0], 'w-', [-1, -1], [line2, -line2], 'w-')

  if ~status, hold off, end	% Return hold to previous status
  return % Suppress output
end
reout = re; 
